<?php

use OptimoleWP\BgOptimizer\Lazyload;
use OptimoleWP\PageProfiler\Profile;
use OptimoleWP\Preload\Links;

/**
 * Class Optml_Manager. Adds hooks for processing tags and urls.
 *
 * @package    \Optml\Inc
 * @author     Optimole <friends@optimole.com>
 */
final class Optml_Manager {
	/**
	 * Holds allowed compatibilities objects.
	 *
	 * @var Optml_compatibility[] Compatibilities objects.
	 */
	public static $loaded_compatibilities = [];
	/**
	 * Cached object instance.
	 *
	 * @var Optml_Manager
	 */
	protected static $instance = null;
	/**
	 * Holds the url replacer class.
	 *
	 * @access  public
	 * @since   1.0.0
	 * @var Optml_Url_Replacer Replacer instance.
	 */
	public $url_replacer;
	/**
	 * Holds the tag replacer class.
	 *
	 * @access  public
	 * @since   1.0.0
	 * @var Optml_Tag_Replacer Replacer instance.
	 */
	public $tag_replacer;
	/**
	 * Holds the lazyload replacer class.
	 *
	 * @access  public
	 * @since   1.0.0
	 * @var Optml_Lazyload_Replacer Replacer instance.
	 */
	public $lazyload_replacer;

	/**
	 * Holds the page profiler class.
	 *
	 * @access  public
	 * @since   1.0.0
	 * @var Profile Page profiler instance.
	 */
	public $page_profiler;

	/**
	 * Holds plugin settings.
	 *
	 * @var Optml_Settings WordPress settings.
	 */
	protected $settings;
	/**
	 * Possible integrations with different plugins.
	 *
	 * @var array Integrations classes.
	 */
	private $possible_compatibilities = [
		'kadence_blocks',
		'essential_blocks',
		'shortcode_ultimate',
		'foogallery',
		'envira',
		'beaver_builder',
		'jet_elements',
		'revslider',
		'metaslider',
		'essential_grid',
		'yith_quick_view',
		'cache_enabler',
		'elementor_builder',
		'divi_builder',
		'thrive',
		'master_slider',
		'pinterest',
		'sg_optimizer',
		'wp_fastest_cache',
		'swift_performance',
		'w3_total_cache',
		'translate_press',
		'give_wp',
		'woocommerce',
		'smart_search_woocommerce',
		'facetwp',
		'wp_rest_cache',
		'wp_bakery',
		'elementor_builder_late',
		'wpml',
		'otter_blocks',
		'spectra',
		'wpsp',
		'jetengine',
		'jetpack',
		'wp_rocket',
		'wp_super_cache',
		'breeze',
		'litespeed_cache',
		'autoptimize_cache',
		'endurance_cache',
		'rocketnet',
		'speedycache',
		'hummingbird',
		'aruba_hsc',
		'spc',
	];
	/**
	 * The current state of the buffer.
	 *
	 * @var boolean Buffer state.
	 */
	private static $ob_started = false;

	/**
	 * Class instance method.
	 *
	 * @codeCoverageIgnore
	 * @static
	 * @return Optml_Manager
	 * @since  1.0.0
	 * @access public
	 */
	public static function instance() {
		if ( null === self::$instance ) {
			self::$instance                    = new self();
			self::$instance->url_replacer      = Optml_Url_Replacer::instance();
			self::$instance->tag_replacer      = Optml_Tag_Replacer::instance();
			self::$instance->lazyload_replacer = Optml_Lazyload_Replacer::instance();
			self::$instance->page_profiler = new Profile();
			add_action( 'after_setup_theme', [ self::$instance, 'init' ] );
			add_action( 'wp_footer', [ self::$instance, 'banner' ] );
		}

		return self::$instance;
	}

	/**
	 * The initialize method.
	 */
	public function init() {

		$this->settings = new Optml_Settings();
		$added_sizes = $this->settings->get( 'defined_image_sizes' );
		foreach ( $added_sizes as $key => $value ) {
			add_image_size( $key, $value['width'], $value['height'], true );
		}
		foreach ( $this->possible_compatibilities as $compatibility_class ) {
			$compatibility_class = 'Optml_' . $compatibility_class;
			$compatibility       = new $compatibility_class();

			/**
			 * Check if we should load compatibility.
			 *
			 * @var Optml_compatibility $compatibility Class to register.
			 */
			if ( $compatibility->will_load() ) {
				if ( $compatibility->should_load_early() ) {
					$compatibility->register();
					continue;
				}
				self::$loaded_compatibilities[ $compatibility_class ] = $compatibility;
			}
		}

		if ( ! $this->should_replace() ) {
			return;
		}
		$this->register_hooks();
	}

	/**
	 * Render banner.
	 *
	 * @return void
	 */
	public function banner() {
		if ( defined( 'REST_REQUEST' ) ) {
			return;
		}

		$has_banner = $this->settings->get( 'banner_frontend' ) === 'enabled';

		if ( ! $has_banner ) {
			return;
		}

		$position       = $this->settings->get( 'badge_position' ) ?? 'right';
		$show_icon_only = $this->settings->get( 'show_badge_icon' ) === 'enabled';

		$string    = __( 'Optimized by Optimole', 'optimole-wp' );
		$div_style = [
			'display'          => 'flex',
			'position'         => 'fixed',
			'align-items'      => 'center',
			'bottom'           => '15px',
			'background-color' => '#fff',
			'padding'          => '8px 6px',
			'font-size'        => '12px',
			'font-weight'      => '600',
			'color'            => '#444',
			'border'           => '1px solid #E9E9E9',
			'z-index'          => '9999999999',
			'border-radius'    => '4px',
			'text-decoration'  => 'none',
			'font-family'      => 'Arial, Helvetica, sans-serif',
		];

		$div_style[ $position ] = '15px';

		$logo = OPTML_URL . 'assets/img/logo.svg';

		$link = tsdk_translate_link( 'https://optimole.com/wordpress/?from=badgeOn' );

		$css = '';

		foreach ( $div_style as $key => $value ) {
			$css .= $key . ':' . $value . ';';
		}

		$output = '<a style="' . esc_attr( $css ) . '" href="' . esc_url( $link ) . '" aria-label="' . esc_html( $string ) . '" rel="nofollow" target="_blank">';
		$output .= '
	<svg style="margin-right:6px" width="20" height="20" viewBox="0 0 251 250" fill="none" xmlns="http://www.w3.org/2000/svg">
<path d="M217.696 99.0926C240.492 152.857 211.284 215.314 153.249 239.999C95.2141 264.683 30.4436 242.239 7.64869 188.474C-15.1462 134.71 14.8767 75.1972 72.9116 50.4991C130.947 25.8007 194.902 45.3142 217.696 99.0926Z" fill="#EDF0FF"/>
<path d="M217.696 99.0926C240.492 152.857 211.284 215.314 153.249 239.999C95.2141 264.683 30.4436 242.239 7.64869 188.474C-15.1462 134.71 14.8767 75.1972 72.9116 50.4991C130.947 25.8007 194.902 45.3142 217.696 99.0926Z" fill="#D7EEFC"/>
<path d="M181.711 75.8034C176.438 78.931 171.418 82.6928 167.832 87.6665C164.246 92.6402 162.221 98.9662 163.346 104.996C163.458 105.602 163.627 106.236 164.091 106.645C164.71 107.18 165.624 107.11 166.425 107.011C178.786 105.334 190.57 99.628 199.556 90.9633C204.956 85.7784 210.103 79.1566 217.542 78.2969C220.987 77.8883 224.475 78.8464 227.582 80.3685C229.186 81.1431 230.732 82.1012 231.899 83.4397C234.571 86.4971 234.796 90.9771 234.402 95.0352C233.967 99.487 232.968 103.926 231.393 108.11C230.001 111.815 226.851 114.661 224.193 117.578C218.203 124.171 211.678 130.301 204.52 135.612C194.437 143.079 184.833 150.772 173.654 156.703C167.93 159.732 162.671 162.029 157.974 166.651C153.488 171.06 149.832 176.33 147.231 182.064C145.093 186.784 143.842 191.546 141.409 196.182C138.836 201.07 135.938 205.804 132.634 210.228C128.584 215.667 124.689 218.809 118.319 221.373C108.63 225.276 98.0689 226.332 87.6486 226.98C79.9569 227.459 72.2365 227.727 64.5448 227.262C62.5477 227.149 58.1321 226.713 57.6544 224.134C57.1197 221.232 61.0852 219.893 63.11 219.471C75.6818 216.85 88.4785 213.145 98.477 205.072C100.502 203.437 102.569 201.056 101.852 198.562C96.5926 198.323 91.5721 200.507 86.5096 201.944C80.9129 203.536 75.0631 204.24 69.2552 204.029C67.3714 203.958 64.9242 203.226 64.9242 201.338C64.9385 199.774 66.6681 198.914 68.0885 198.266C72.9537 196.083 77.0602 192.56 81.096 189.08C84.5268 186.122 88.1269 182.909 89.3643 178.541C89.533 177.95 89.6457 177.288 89.3643 176.752C88.7456 175.541 86.9597 175.865 85.7084 176.386C79.3382 179.021 73.1085 182.007 67.0617 185.332C59.4121 189.545 51.3682 194.448 42.678 193.73C40.9623 193.589 38.7826 192.575 39.12 190.884C39.303 189.94 40.2308 189.348 41.0607 188.869C63.2649 175.935 82.277 156.238 87.3534 131.019C91.5019 110.406 85.9334 89.0332 87.8178 68.0968C88.0987 64.9266 88.5631 61.7426 89.6458 58.7416C90.3911 56.6705 91.4173 54.6978 92.5287 52.7957C99.8831 40.0872 110.711 29.4076 123.493 22.208C135.291 15.5862 148.328 15.0649 160.829 9.92232C168.788 6.6536 176.874 2.80726 185.48 2.97633C186.071 2.99042 186.689 3.03269 187.153 3.38492C187.561 3.70897 187.772 4.21618 187.941 4.72339C189.263 8.6261 189.249 12.8529 190.261 16.8542C190.36 17.2628 190.514 17.7136 190.88 17.9109C191.245 18.1081 191.71 17.9954 192.103 17.8686C195.478 16.7978 198.853 15.727 202.228 14.6563C203.283 14.3181 204.422 13.9941 205.448 14.3745C207.431 15.1071 207.656 17.7981 207.459 19.9116C205.533 41.0312 199.936 64.9968 181.711 75.8034Z" fill="#577BF9"/>
<path d="M118.165 39.6223C108.461 45.8356 99.8553 53.7539 92.8244 62.8836C97.6475 64.3067 102.4 66.0112 107.027 67.9977C103.132 71.8867 98.4348 74.9718 93.3303 77.0008C87.6496 79.2549 83.7963 82.0873 80.3649 87.1594C87.6772 89.1315 95.2572 85.9895 102.387 83.4258C109.572 80.8472 116.941 78.8043 124.436 77.3248C129.006 76.423 133.858 75.6341 137.429 72.6331C137.711 72.3937 138.006 72.0835 137.992 71.7175C137.978 71.4073 137.739 71.1541 137.514 70.9424C134.195 67.885 129.47 66.6172 126.391 63.306C126.053 62.94 125.716 62.4607 125.843 61.9819C125.941 61.6015 126.306 61.3477 126.644 61.1365C130.005 59.009 133.352 56.8816 136.712 54.7541C132.41 50.1187 128.036 45.5397 123.593 41.0453C122.242 39.6786 119.781 38.5796 118.165 39.6223Z" fill="#EE686B"/>
<path d="M165.793 91.1041C163.67 96.2326 163.205 101.953 163.74 107.49C168.296 107.011 172.782 105.897 177.015 104.165C177.703 103.883 178.435 103.545 178.815 102.897C179.067 102.46 179.124 101.924 179.152 101.417C179.461 95.4293 177.085 89.3432 178.617 83.5524C178.969 82.2277 180.08 78.7904 177.788 78.4095C176.551 78.1983 174.553 80.4385 173.639 81.1291C170.152 83.7779 167.466 87.0465 165.793 91.1041Z" fill="#F6E5C1"/>
<path d="M171.558 156.534C169.041 157.929 166.524 159.338 164.274 161.128C161.222 163.565 158.762 166.665 156.498 169.863C150.17 178.824 145.276 188.7 140.411 198.534C139.44 200.479 138.484 202.423 137.514 204.368C136.318 206.776 135.109 209.242 134.8 211.919C134.687 212.905 134.771 214.061 135.573 214.652C136.108 215.047 136.839 215.075 137.499 215.033C140.298 214.892 143.012 214.061 145.656 213.131C164.893 206.34 181.81 193.138 193.073 176.104C194.283 174.273 195.45 172.385 196.223 170.328C196.913 168.482 197.292 166.552 197.63 164.607C198.572 159.226 199.275 153.801 199.697 148.349C199.838 146.587 199.964 142.488 197.574 142.192C195.563 141.938 190.782 145.827 188.967 146.827C183.174 150.068 177.366 153.308 171.558 156.534Z" fill="#F6E5C1"/>
<path d="M58.9198 140.29C55.4043 141.952 51.6495 143.798 49.7369 147.194C47.7403 150.744 48.2744 155.365 50.4259 158.831C52.5778 162.297 56.1071 164.72 59.862 166.298C60.9728 166.763 62.1538 167.172 63.3492 167.031C64.6287 166.89 65.7539 166.143 66.7807 165.354C71.59 161.607 74.9644 156.351 78.241 151.223C82.2209 145.01 86.2987 138.486 87.1005 131.146C87.8878 123.862 79.7317 130.498 77.102 131.738C71.0415 134.583 64.9804 137.43 58.9198 140.29Z" fill="#336093"/>
<path d="M136.515 18.9961C142.604 15.6429 149.129 13.149 155.626 10.6694C164.471 7.30208 173.358 3.92065 182.667 2.24404C183.427 2.10315 184.228 1.99043 184.931 2.2863C185.663 2.58218 186.155 3.27255 186.577 3.93474C188.995 7.82333 190.163 12.4869 189.853 17.0659C189.825 17.5167 189.769 17.9957 189.474 18.3339C189.277 18.5734 188.981 18.7002 188.7 18.827C185.747 20.1655 182.653 21.2645 179.827 22.8565C176.986 24.4486 174.357 26.6183 172.923 29.5348C170.265 34.9309 172.135 41.7078 169.618 47.1744C168.183 50.3022 165.469 52.627 162.727 54.6982C158.621 57.812 153.756 60.714 148.651 60.0097C145.698 59.6011 143.04 58.0232 140.495 56.4591C139.103 55.6137 137.724 54.7827 136.332 53.9374C130.088 50.1473 123.353 45.7373 121.286 38.7209C119.725 33.3952 121.286 30.7183 125.068 27.1396C128.5 23.9273 132.409 21.2504 136.515 18.9961Z" fill="#D6EDFB"/>
<path d="M199.416 58.9386C199.894 59.8829 200.78 60.7847 201.835 60.6575C202.285 60.6011 202.693 60.3761 203.086 60.1362C204.605 59.235 206.18 58.2907 207.08 56.7691C207.924 55.3604 208.064 53.6553 208.177 52.0351C208.416 48.6115 208.5 44.8355 206.391 42.1304C206.152 41.8346 205.857 41.5246 205.477 41.4823C202.468 41.102 200.682 45.7232 200.161 47.8648C199.275 51.4434 197.574 55.3322 199.416 58.9386Z" fill="#F6E5C1"/>
<path d="M220.565 71.295C217.767 72.5633 214.673 74.9163 215.193 77.9594C218.892 77.7764 222.647 77.8046 226.19 78.9031C229.72 80.0023 233.024 82.3127 234.571 85.6803C236.99 90.9354 234.656 97.163 235.696 102.841C240.646 100.178 244.626 95.7538 246.763 90.555C247.663 88.343 248.254 85.8915 247.733 83.5667C247.298 81.6366 246.13 79.8895 245.891 77.9312C245.638 75.8319 246.468 73.789 247.199 71.8025C247.523 70.9428 247.79 69.8581 247.171 69.1675C246.552 68.4775 245.47 68.6185 244.569 68.7733C240.28 69.5341 236.146 69.2665 231.885 68.6041C228.089 68.0264 224.067 69.7171 220.565 71.295Z" fill="#F6E5C1"/>
<path d="M40.5681 149.138C40.4415 149.8 40.3994 150.49 40.1606 151.124C39.1338 153.773 35.2808 154.125 33.7621 156.52C33.4246 157.056 33.2137 157.704 33.3121 158.338C33.495 159.578 34.7324 160.381 35.9277 160.79C37.123 161.198 38.4167 161.409 39.387 162.212C40.9341 163.494 41.1166 165.876 42.5088 167.341C43.451 168.341 44.8289 168.778 46.1509 169.102C49.0477 169.82 52.029 170.201 55.0103 170.229C55.8397 170.243 56.7681 170.173 57.3166 169.553C58.0055 168.778 57.682 167.482 56.9649 166.735C56.2478 165.988 55.2492 165.594 54.3352 165.087C52.0428 163.818 50.2292 161.705 49.3147 159.239C48.3587 156.619 48.4571 153.745 49.1604 151.068C49.6382 149.264 52.5631 145.235 51.2416 143.53C50.0601 142.008 46.4605 143.009 45.0821 143.643C42.8743 144.672 41.0464 146.686 40.5681 149.138Z" fill="#F6E5C1"/>
<path d="M25.8872 210.694C26.0419 211.68 27.1809 212.116 28.1371 212.37C30.6543 213.018 33.1855 213.554 35.7448 214.019C36.9823 214.23 38.2338 214.427 39.4715 214.23C40.7089 214.033 41.932 213.37 42.5087 212.272C42.8321 211.651 42.9449 210.933 42.9726 210.214C43.0151 209.242 42.9023 208.242 42.4384 207.383C41.5665 205.79 39.6257 205.086 37.812 205.044C36.012 205.016 34.2401 205.509 32.4402 205.748C30.134 206.058 25.3247 207.242 25.8872 210.694Z" fill="#6B8FCA"/>
<path d="M151.745 39.27C154.262 38.2133 157.186 38.2133 159.816 39.4954C161.026 38.2274 162.165 36.8748 163.219 35.48C163.304 35.3673 164.57 33.5639 164.471 33.578C163.29 33.6766 162.109 33.9584 160.998 34.381C160.351 34.6206 159.577 33.9725 158.959 33.7471C158.129 33.4371 157.271 33.2257 156.399 33.113C154.599 32.8876 152.392 33.099 150.746 33.8738C148.721 34.8319 146.935 36.3394 145.599 38.1147C143.631 40.7212 142.548 43.9617 143.11 47.2022C143.153 47.4276 143.251 47.6953 143.476 47.7094C143.617 47.7094 143.729 47.6249 143.827 47.5263C145.332 46.1596 146.373 44.328 147.512 42.6655C148.623 41.1157 150.099 39.9604 151.745 39.27Z" fill="#1D445C"/>
<path opacity="0.5" d="M132.213 48.3997C132.606 48.8223 133.07 49.2732 133.605 49.8226C135.475 51.7106 137.022 53.8942 138.991 55.6695C140.284 56.825 142.028 58.121 143.814 58.3045C144.404 58.3609 148.567 58.4875 148.68 58.1635C151.45 49.4423 153.812 40.6788 156.287 31.8449C156.962 29.4356 164.232 9.25987 162.446 8.69632C161.997 8.55544 161.504 8.75266 161.068 8.93582C156.147 11.0915 151.239 13.2471 146.317 15.4028C142.942 16.8821 142.155 18.8828 140.72 22.0811C138.681 26.6459 136.727 31.239 134.842 35.8603C133.717 38.6217 132.649 41.4114 131.537 44.187C130.694 46.2158 131.116 47.188 132.213 48.3997Z" fill="white"/>
<path opacity="0.5" d="M157.693 55.6844C161.448 54.2187 165.498 52.5142 167.453 48.9778C168.507 47.0616 168.817 44.8496 169.098 42.6799C170.799 29.8165 172.501 16.939 174.203 4.07556C174.258 3.61062 170.687 4.97727 170.335 5.35768C169.407 6.3439 169.393 7.59788 169.028 8.87995C167.973 12.5995 166.693 16.2627 165.625 19.9822C163.473 27.4918 161.616 35.0859 160.042 42.7363C159.184 47.0335 158.396 51.3588 157.693 55.6844Z" fill="white"/>
<path d="M199.289 106.546C190.261 110.969 181.725 112.703 171.839 112.914C164.077 113.083 155.879 112.351 149.481 107.941C147.948 106.884 146.556 105.644 145.346 104.249C148.44 104.63 151.52 106.475 154.529 107.349C157.623 108.25 160.829 108.73 164.049 108.405C170.785 107.729 177.267 105.292 183.188 102.094C189.361 98.7685 195.042 94.6402 200.737 90.5544C202.551 89.2584 204.225 87.8635 205.589 86.1446C207.122 84.2145 208.823 82.4674 211.073 81.3965C213.956 80.0298 217.275 79.9452 220.467 80.0718C223.617 80.199 226.865 80.5512 229.579 82.1573C233.995 84.7636 235.978 90.7236 234.121 95.4291C232.883 92.8229 229.973 91.2167 227.09 91.2449C222.59 91.287 217.809 95.4573 214.209 97.7965C209.385 100.938 204.478 103.953 199.317 106.489C199.331 106.531 199.303 106.531 199.289 106.546Z" fill="#6C99CE"/>
<path d="M179.096 24.4344C173.091 28.0553 171.249 34.4096 171.46 41.0033C171.685 47.8084 176.649 52.4015 183.23 54.1485C203.058 59.4036 207.361 34.4519 208.05 20.1513C208.106 18.8551 208.078 17.3898 207.15 16.474C206.503 15.8401 205.575 15.6287 204.675 15.516C198.994 14.8115 192.792 16.474 188.081 19.6864C185.241 21.6166 181.964 22.7015 179.096 24.4344Z" fill="#6C99CE"/>
<path d="M86.9743 69.4919C86.5242 72.7041 86.3556 76.1137 87.7473 79.044C93.9488 75.7897 99.9397 72.1263 105.663 68.0826C102.119 67.0963 98.5756 66.1105 95.032 65.1098C94.4973 64.9688 93.9206 64.7858 93.6397 64.307C93.3445 63.8277 93.4567 63.194 93.6254 62.6583C94.4409 60.052 96.3115 57.9246 98.224 55.9663C103.764 50.2881 110.036 45.3146 116.828 41.2288C118.192 40.4115 119.655 39.524 120.26 38.0728C121.202 35.8326 119.796 33.0288 121.005 30.9295C121.806 29.5347 123.452 28.9571 124.872 28.1822C127.656 26.6746 128.331 23.3918 124.829 22.2224C122.074 21.3066 119.936 23.6595 117.77 25.0403C112.427 28.4499 107.449 31.8594 103.075 36.5088C94.5537 45.54 88.7038 57.1638 86.9743 69.4919Z" fill="#336093"/>
<path d="M94.9195 86.8914C88.4791 87.3143 86.6794 88.0751 87.1576 94.5139C87.7056 102.023 88.4227 109.589 88.4791 117.127C88.5494 126.341 87.3821 135.795 83.1213 143.967C79.9437 150.053 75.2605 155.238 71.1407 160.677C67.1608 165.932 63.3639 170.99 58.6674 175.696C61.6625 172.695 68.328 171.469 72.111 169.313C76.7655 166.665 81.1386 163.522 85.1467 159.958C100.123 146.658 112.005 129.314 115.422 109.265C117.053 99.6988 116.857 89.9206 116.66 80.2275C109.826 82.5237 102.133 86.4264 94.9195 86.8914Z" fill="#336093"/>
<path d="M43.1137 204.283C42.9169 204.156 42.7062 204.015 42.5094 203.93C41.0746 203.184 39.6685 203.282 38.0093 203.297C35.9984 203.325 34.0297 203.423 32.0609 203.776C28.1798 204.452 24.383 205.79 20.8674 207.396C18.8143 208.341 12.0785 210.679 15.7488 213.666C17.1128 214.793 18.8705 215.286 20.6002 215.653C25.1142 216.653 29.7688 216.991 34.3812 216.681C36.8562 216.512 38.8391 216.132 41.131 215.061C41.9891 214.652 42.8605 214.244 43.5638 213.61C46.2919 211.186 46.1371 206.382 43.1137 204.283ZM40.5686 209.862C38.9514 212.328 33.889 211.595 31.5125 211.37C30.711 211.285 29.8672 211.158 29.3188 210.567C29.2063 210.44 29.1079 210.271 29.1079 210.116C29.1079 209.806 29.4032 209.608 29.6844 209.482C31.4 208.622 33.2844 208.059 35.1968 207.847C36.4624 207.72 38.6983 207.312 39.8233 208.059C40.8075 208.721 40.9203 209.327 40.5686 209.862Z" fill="#1D445C"/>
<path d="M185.1 49.0198C186.155 48.2309 187.336 47.2164 187.238 45.9061C187.111 44.3282 185.1 43.4687 183.722 43.8773C182.372 44.2718 180.966 45.8498 180.052 46.8501C175.453 51.894 170.869 56.9519 166.271 61.9958C165.933 62.3624 164.302 63.7993 165.708 63.7429C166.679 63.7009 167.986 62.5034 168.746 61.9538C170.743 60.4887 172.529 58.7554 174.455 57.2195C177.943 54.4302 181.5 51.6968 185.1 49.0198Z" fill="#1D445C"/>
<path d="M175.411 66.871C175.017 67.5334 173.302 70.1258 175.271 68.7171C176.775 67.618 177.802 66.0821 178.983 64.6733C180.234 63.18 181.514 61.7005 182.836 60.2636C184.074 58.9107 185.48 57.7414 186.703 56.3608C187.772 55.1489 190.233 52.4158 188.475 50.7533C187.913 50.2179 186.703 50.6828 186.141 51.0632C185.142 51.7395 184.748 52.8384 184.059 53.7822C183.3 54.8249 182.695 55.9522 181.936 56.9806C179.63 60.179 177.45 63.4758 175.411 66.871Z" fill="#1D445C"/>
<path d="M248.746 78.0012C248.605 77.7336 248.451 77.508 248.296 77.325C248.914 75.8035 249.407 74.1692 249.744 72.3801C249.871 71.7316 250.18 70.21 249.463 68.7449C248.83 67.491 247.55 66.5749 246.06 66.2791C245.005 66.0817 244.007 66.1945 243.093 66.3637C242.137 66.5328 241.222 66.7722 240.323 67.068C237.046 65.1241 233.615 65.5045 230.563 65.8423L229.917 65.9269C226.359 66.3355 222.843 67.5612 219.75 69.5057C215.77 71.9715 212.226 75.5923 209.189 80.2697C206.067 82.3407 203.241 84.9614 200.583 87.399C199.05 88.8221 197.601 90.1601 196.153 91.372C192.778 94.1336 189.727 96.6133 186.338 98.6844C184.903 99.5861 183.398 100.403 181.852 101.122C179.756 94.373 179.883 87.1596 182.274 81.397C183.469 78.579 185.269 76.4377 187.195 74.1548C188.081 73.1126 188.981 72.0135 189.895 70.8016C190.5 69.9845 191.105 69.1535 191.723 68.3081C193.327 66.0817 194.803 63.7856 196.223 61.4464C196.8 60.4888 197.826 59.5306 198.684 60.9394C199.092 61.5874 199.205 62.4189 199.753 62.9541C200.681 63.7994 202.144 63.3063 203.254 62.7147C205.715 61.3762 208.106 59.5163 209.076 56.8536C209.793 54.8671 209.611 52.6549 209.414 50.5416C209.104 47.1884 208.795 43.8492 208.5 40.5242C208.289 38.2981 207.909 36.072 208.148 33.8459C208.373 31.5071 209.245 29.1824 209.709 26.8718C209.821 26.3363 209.99 25.6742 210.159 24.9556C210.651 22.969 211.143 20.9684 211.607 18.9959C212.086 16.9389 212.901 14.9523 213.435 12.9939C214.139 10.4579 214.87 7.1328 213.323 4.77991C211.874 2.582 208.682 1.8071 206.573 3.44144C206.404 3.55415 206.278 3.68096 206.137 3.82185C205.589 4.3995 205.153 5.13215 204.422 5.44209C203.831 5.7098 203.17 5.65346 202.509 5.62526C200.892 5.59707 199.219 5.89297 197.925 6.86509C195.267 8.89393 195.998 11.2891 197.756 13.318C197.32 13.6279 196.912 13.9097 196.561 14.0647C195.605 14.5578 194.634 14.9523 193.65 15.2764C192.919 12.8389 192.173 10.4156 191.414 7.96405C190.669 5.56892 189.867 1.34215 187.322 0.271376C186.689 0.00368107 185.972 -0.0104081 185.283 0.00368111C176.255 0.0882161 167.733 3.61051 159.577 7.17509C157.721 7.97815 155.893 8.73896 154.107 9.49978C150.985 10.8101 147.877 12.1204 144.812 13.5575C143.546 14.1492 142.407 14.9382 141.071 15.2905C139.707 15.6567 138.287 15.7695 136.909 16.1076C135.531 16.4598 134.167 16.9389 132.774 17.277C130.426 17.8265 128.092 18.531 125.828 19.3622C121.342 21.0106 117.039 23.1663 113.059 25.7728C105.114 30.9435 98.3362 37.8754 93.3157 45.9344C90.2083 50.922 87.7893 56.3461 86.1438 61.9959C85.3426 64.7719 84.5409 67.5612 84.0488 70.4074C83.5423 73.3238 83.5705 76.1275 83.7535 79.0721C83.7535 79.1705 83.7673 79.2551 83.7392 79.3541C83.7115 79.4669 83.6407 79.5653 83.5705 79.6637C82.7268 80.8192 81.4755 81.5236 80.4349 82.5099C77.594 85.1726 74.5004 87.5118 72.1101 90.6251C72.124 90.5969 74.1349 90.6958 74.3318 90.7097C75.1191 90.724 75.9065 90.7379 76.6944 90.7522C77.4818 90.7522 78.2691 90.7522 79.0565 90.7522C79.8444 90.7379 80.6317 90.724 81.4191 90.7097C82.291 90.6815 83.1486 90.5692 84.0067 90.4985C84.1754 90.4846 84.3722 90.4846 84.4989 90.6112C84.5973 90.7097 84.6111 90.8506 84.6393 90.9778C84.9346 93.2319 85.6379 95.4721 85.6379 97.7964C85.6661 98.1492 85.7225 98.5014 85.7363 98.8535C85.9331 100.53 86.0736 102.235 86.1582 103.926C86.2002 104.362 86.2428 104.813 86.2566 105.25C86.3412 106.701 86.3832 108.152 86.397 109.631C86.4396 111.576 86.4252 113.506 86.3268 115.45C86.2146 118.367 85.947 121.298 85.5251 124.2C79.1411 128.201 71.7867 132.047 62.3931 136.359C60.9445 137.007 59.4118 137.725 57.8647 138.542C57.5413 138.641 57.1338 138.726 56.7539 138.782C55.7133 138.994 54.6727 139.163 53.6603 139.303C48.9494 140.05 44.0981 140.839 39.7809 144.531C37.2074 146.715 35.6043 149.504 35.2809 152.378C35.2668 152.547 35.2809 152.702 35.2809 152.871C34.2403 153.618 33.2278 154.491 32.2294 155.52C31.8216 155.943 30.8654 156.929 30.6966 158.394C30.556 159.691 31.0482 161.029 32.0607 161.987C32.7497 162.635 33.5232 163.016 34.2544 163.34C34.9434 163.635 35.6184 163.861 36.3215 164.058C37.7699 166.848 40.3151 168.13 42.5654 169.229L43.0432 169.468C47.4449 171.652 51.9866 171.708 56.3884 171.751C56.9226 171.751 57.429 171.765 57.9493 171.765C57.8365 171.878 57.7243 171.99 57.5977 172.103C52.4793 176.654 46.8821 180.74 40.9763 184.234C35.309 187.615 29.3045 190.067 23.2436 192.673C22.7515 192.885 22.2171 193.152 22.0343 193.659C21.7671 194.392 22.414 195.125 23.0468 195.547C24.2139 196.35 25.7045 196.984 27.0685 197.337C28.2779 197.647 29.5295 197.675 30.7529 197.816C32.9888 198.055 35.2809 198.027 37.5309 197.901C45.9399 197.407 54.462 194.646 62.0415 191.011C63.8976 190.123 65.7256 189.179 67.5535 188.249C69.6071 187.192 71.7303 186.094 73.8397 185.121C74.2754 184.91 74.8239 184.628 75.4426 184.319C77.0599 183.515 78.6346 182.67 80.2381 181.825C80.927 181.472 81.5739 181.064 82.249 180.697C82.938 180.317 83.894 179.584 84.6813 179.528C85.7501 179.443 84.5973 181.965 84.344 182.444C83.7955 183.515 82.98 184.417 82.1224 185.262C81.841 185.558 81.5457 185.84 81.2643 186.108C79.6614 187.615 77.8893 188.926 76.0613 190.151C74.444 191.222 72.7847 192.222 71.0972 193.152C69.5789 193.998 68.2148 194.913 66.7805 195.815C64.868 197.013 53.4635 201.282 55.9383 204.551C56.7683 205.65 58.7228 206.1 59.9603 206.382C65.6974 207.721 71.7303 206.706 77.3833 205.452C83.3316 204.128 89.1251 202.198 94.975 200.549C95.5656 200.38 96.3816 200.324 96.6205 200.873C98.1394 204.339 86.5518 208.975 84.527 209.904C77.046 213.342 69.2272 215.723 61.1557 217.386C57.9913 218.034 54.8132 218.696 51.6212 219.161C48.8367 219.569 45.659 219.344 43.1416 220.781C39.7389 222.725 43.7183 225.825 45.8133 226.952C48.2179 228.249 50.946 228.84 53.6039 229.432C63.1385 231.531 73.1928 231.08 82.8677 230.418C101.978 229.08 120.849 225.008 139.102 219.231C141.015 218.626 142.913 218.005 144.826 217.372C146.851 216.681 148.946 216.005 150.985 215.202C157.538 212.666 163.543 209.51 169.013 205.762C171.094 204.339 173.105 202.817 175.032 201.226C176.283 200.197 177.492 199.126 178.659 198.013C180.094 196.675 181.458 195.266 182.794 193.814C189.91 186.023 195.436 176.781 199.205 166.284C201.961 158.578 202.607 151.406 201.44 141.628C207.585 137.007 213.45 131.991 218.976 126.538C224.348 121.213 230.381 115.211 234.191 107.405L234.318 107.448C234.529 106.898 235.162 106.293 235.767 105.715C236.709 104.813 237.679 103.954 238.621 103.108C242.91 99.2339 247.34 95.2327 249.294 88.7093C250.462 84.863 250.25 81.073 248.746 78.0012ZM222.323 73.6478C224.812 72.0981 227.624 71.1118 230.465 70.7878L231.112 70.7032C231.449 70.675 231.801 70.633 232.125 70.5904C231.73 70.9006 231.379 71.2384 231.013 71.6049C230.296 72.3237 229.509 73.0563 228.651 73.8027C228.383 74 228.116 74.2112 227.849 74.4086C226.907 75.1273 225.937 75.8317 224.995 76.5361C223.265 76.1839 221.479 76.0429 219.665 76.2259C219.356 76.2403 219.075 76.2967 218.779 76.3387C219.918 75.3103 221.099 74.4086 222.323 73.6478ZM203.93 57.9245C203.662 58.6709 202.045 60.3334 201.188 59.4599C200.99 59.2487 200.892 58.9529 200.808 58.6709C200.414 57.3606 199.739 55.9801 200.006 54.6554C200.175 53.8664 201.243 50.3584 202.157 50.4429C203.691 50.5698 204.323 56.7829 203.93 57.9245ZM206.77 49.6821C206.798 50.3584 206.812 50.9361 206.404 51.5138C205.434 52.8804 204.914 49.8512 204.787 49.3017C204.436 47.8364 204.408 46.2866 205.055 44.9059C205.406 44.1169 206.081 44.4973 206.278 45.2018C206.658 46.6389 206.728 48.1887 206.77 49.6821ZM136.149 23.913C142.519 19.6299 149.073 16.8825 156.033 13.966C157.848 13.2052 159.704 12.4304 161.56 11.6132C168.422 8.62628 175.13 6.11837 182.498 4.52631C183.118 4.3995 183.75 4.25861 184.369 4.42768C185.325 4.70947 185.719 6.09023 186.113 6.92148C186.647 8.07678 186.802 9.373 187.153 10.5987C187.8 12.698 188.7 14.6846 189.066 16.8543C189.08 16.953 189.094 17.0516 189.038 17.1361C189.009 17.1784 188.939 17.2207 188.869 17.2489C187.392 17.8828 185.902 18.545 184.425 19.1791C183.82 19.4468 183.089 19.9258 182.414 19.9821C181.866 20.0103 181.261 19.8694 180.685 19.8835C176.691 19.9821 174.624 22.5887 172.472 25.5051C170.026 28.7738 169.632 32.7892 169.35 36.7342C169.041 40.9891 169.027 44.3564 165.962 47.7801C163.557 50.4852 160.337 52.3309 157.13 53.9936C154.234 55.5151 151.168 56.952 147.892 56.9802C142.449 57.0792 138.864 52.6549 135.137 49.3863C132.31 46.9066 130.004 43.7506 127.515 40.9046C125.87 39.0307 123.001 36.4524 122.945 33.7613C122.917 32.5356 123.873 31.7466 124.745 31.0281C126.798 29.3796 129.09 28.0411 131.383 26.7449C132.971 25.8714 134.63 24.9415 136.149 23.913ZM92.5284 57.9102C94.4127 53.1902 97.1828 48.5409 100.234 44.4409C103.441 40.1156 107.238 36.2129 111.484 32.8878C113.622 31.2253 115.858 29.7037 118.192 28.337C119.893 27.3508 121.468 26.3223 123.24 25.491C124.126 25.0824 125.026 24.6597 125.982 24.5752C123.409 26.3786 120.794 28.1398 118.291 30.0137C117.657 30.4927 116.997 30.9999 116.659 31.6903C116.322 32.4511 116.378 33.3528 116.659 34.1559C117.292 35.917 118.909 37.0583 116.856 38.5376C113.368 41.0173 109.895 43.4829 106.408 45.9626C103.638 47.9351 101.106 50.288 98.6315 52.6409C97.5207 53.6696 96.3955 54.6836 95.4251 55.8391C94.2861 57.1914 93.3721 58.8965 91.9235 59.9392C92.0362 59.1359 92.2469 58.6289 92.5284 57.9102ZM88.7878 74.8033C88.816 74.3948 88.8581 73.9718 88.9001 73.5632C88.9847 72.7605 89.0831 71.9571 89.1954 71.1682C89.3363 70.1818 89.4486 69.1955 89.6034 68.2235C89.6316 68.0262 89.9551 65.6311 90.1939 65.6875C91.0238 65.9131 91.8676 66.1525 92.7114 66.3216C94.1456 66.6174 95.5517 66.9978 96.9722 67.3356C98.35 67.674 99.7423 67.9134 101.12 68.2374C101.556 68.3363 101.992 68.4491 102.428 68.5475C102.456 68.5475 101.852 68.9423 101.81 68.9705C101.486 69.1391 101.163 69.2945 100.839 69.4493C100.192 69.7733 99.5311 70.0834 98.8847 70.4074C97.6047 71.0272 96.3252 71.6613 95.0734 72.3657C92.922 73.5351 90.8408 74.7469 88.7176 75.9865C88.8017 75.9163 88.7735 74.9438 88.7878 74.8033ZM85.8209 85.187C85.4831 85.2151 85.1458 85.2572 84.8079 85.2854C84.6255 85.2997 84.4568 85.3279 84.2738 85.3418C84.26 85.3418 83.9365 85.3561 83.9365 85.3279C84.2456 84.8768 84.7377 84.5954 85.174 84.257C85.5533 83.9612 85.8767 83.5947 86.2284 83.285C90.1801 79.7201 95.2001 77.7054 99.8125 75.2119C103.905 72.9855 107.87 70.5058 111.653 67.7724C111.836 67.632 112.033 67.5048 112.216 67.3638C111.695 67.1106 111.048 67.026 110.486 66.8286C109.783 66.5749 109.066 66.3498 108.348 66.1381C106.858 65.7013 105.367 65.3209 103.863 64.9549C102.4 64.6027 100.938 64.2506 99.4891 63.842C99.1092 63.743 95.2283 62.4045 95.2001 62.4471C95.8049 61.4607 96.5923 60.6436 97.394 59.8264C98.6171 58.6007 99.8827 57.4314 101.12 56.2195C102.597 54.7682 104.242 53.5142 105.803 52.1477C110.542 47.9914 115.661 44.5959 120.906 41.1582C122.354 43.1448 124.014 44.9764 125.673 46.7939C127.586 48.8932 129.484 50.9924 131.396 53.0776C131.875 53.6132 132.423 54.1909 132.999 54.7543C132.62 54.9799 132.24 55.2049 131.875 55.4023C127.586 58.2623 123.17 60.9537 118.656 63.4334C118.726 63.3908 119.19 63.6446 119.275 63.6866C119.485 63.7712 119.683 63.8558 119.893 63.9404C122.439 65.0252 124.913 66.2791 127.361 67.5756C128.584 68.2235 129.793 68.8859 131.017 69.5339C131.635 69.8579 132.24 70.1818 132.859 70.5058C133.421 70.8016 134.026 71.0416 134.532 71.464C134.462 71.5204 134.363 71.5624 134.293 71.6049C134.195 71.6752 134.096 71.7459 133.998 71.8023C133.815 71.9151 133.618 72.0274 133.407 72.1263C133.056 72.2955 132.69 72.4359 132.325 72.5631C131.453 72.8871 130.581 73.2111 129.709 73.5351C128.893 73.8308 128.12 74.1974 127.29 74.4794C126.896 74.606 126.559 74.7188 126.306 74.8172C121.806 76.4797 117.039 78.0294 111.695 79.5791C111.203 79.7201 110.725 79.8611 110.233 80.0021C102.316 82.3551 94.0892 84.3836 85.8209 85.187ZM42.9868 148.222C44.8711 146.587 46.9805 145.728 49.2585 145.122C48.9494 145.517 48.6398 145.869 48.3589 146.292C47.9652 146.898 47.67 147.588 47.4167 148.321C47.2337 148.334 47.0651 148.363 46.8821 148.391C45.5744 148.504 44.2247 148.616 42.8883 148.912C42.5931 148.982 42.3404 149.039 42.059 149.123C42.326 148.799 42.6356 148.518 42.9868 148.222ZM36.209 158.803C36.1668 158.774 36.0825 158.732 35.984 158.69C38.0793 156.591 40.2028 155.337 42.7899 154.689L46.6571 153.731C46.7416 155.83 47.1497 157.929 47.8104 159.691C47.5854 159.719 47.3742 159.761 47.1353 159.789C43.2262 160.057 39.5277 160.198 36.209 158.803ZM45.293 165.072C46.0383 164.904 46.8539 164.763 47.6976 164.65C47.9652 164.636 48.2179 164.622 48.4855 164.608C49.2867 164.537 50.0885 164.495 50.8902 164.439C51.7898 165.326 52.8448 166.115 54.012 166.791C50.9184 166.693 47.9791 166.411 45.293 165.072ZM57.9211 163.269C55.4883 162.24 53.646 160.691 52.7602 158.873C51.4243 156.21 51.0024 151.195 52.4511 148.969C54.1524 146.391 56.9369 144.531 59.9459 142.938C60.2694 142.812 60.579 142.643 60.9025 142.445C62.1117 141.839 63.3071 141.29 64.5025 140.726C72.2644 137.19 78.649 133.921 84.3158 130.61C83.5567 133.738 82.5581 136.866 81.2786 139.937C77.594 148.743 71.9691 157.281 64.8254 165.016C62.3931 164.608 60.0587 164.171 57.9211 163.269ZM194.62 164.622C193.987 166.439 193.27 168.229 192.483 169.99C192.188 170.68 191.878 171.342 191.569 172.004C187.842 179.81 182.752 186.925 176.648 193.011C175.791 193.871 174.905 194.73 173.991 195.547C172.416 196.984 170.785 198.351 169.083 199.648C168.887 199.816 168.69 199.957 168.492 200.098C167.143 201.113 165.765 202.057 164.386 202.944C158.551 206.635 152.349 209.129 145.487 210.877C144.052 211.257 139.314 212.891 138.371 210.975C138.16 210.524 138.188 209.96 138.287 209.453C138.962 206.016 141.493 202.789 142.998 199.661C147.947 189.362 150.521 178.274 158.705 169.82C163.838 164.509 170.236 160.437 176.691 157.14C183.694 153.562 190.444 149.49 196.884 144.967C197.531 152.547 196.842 158.408 194.62 164.622ZM232.645 95.5988C232.349 98.5014 231.604 101.347 230.423 104.024C227.048 111.675 220.945 117.719 215.545 123.045C203.367 135.076 189.558 145.066 174.469 152.772C167.986 156.07 161.419 159.676 156.118 164.763C148.552 171.99 144.967 180.162 141.352 189.743C138.681 196.773 134.617 203.282 129.54 208.834C121.412 217.71 109.811 222.303 98.2234 224.487C92.2469 225.614 86.1438 226.149 80.0694 226.318C74.9643 226.445 67.4695 227.46 62.8006 225.05C62.6038 224.952 62.3788 224.839 62.1963 224.712C62.1819 224.698 62.1681 224.698 62.1537 224.684C61.8585 224.472 61.5776 224.191 61.5212 223.853C61.3382 223.035 62.2521 222.444 63.0257 222.148C70.7179 219.189 78.9022 217.738 86.4252 214.314C91.741 211.919 96.6205 208.566 100.698 204.381C102.316 202.719 110.205 193.829 105.086 192.236C104.664 192.11 104.2 192.236 103.778 192.363C98.7299 193.97 93.8222 195.843 88.7033 197.238C85.5251 198.126 82.3474 198.985 79.1693 199.873C76.4832 200.605 73.6146 201.211 70.8445 201.324C70.5349 201.338 70.2114 201.352 69.9162 201.254C67.4131 200.408 72.3772 197.999 72.8975 197.717C75.1191 196.449 77.3833 195.224 79.4784 193.744C83.3598 191.039 86.6503 187.714 89.4768 183.938C91.3893 181.36 93.3019 178.556 93.9062 175.315C94.1313 174.146 94.708 170.173 93.3859 169.37C93.1471 169.243 92.8518 169.257 92.5704 169.285C90.9531 169.553 89.5896 170.596 88.2394 171.511C83.5285 174.738 78.3111 177.372 73.2491 179.951C72.7006 180.233 72.1942 180.486 71.8005 180.669C69.6071 181.698 67.4551 182.811 65.3463 183.882C59.1586 187.066 52.6899 189.7 45.659 190.236C45.2089 190.264 44.745 190.306 44.337 190.109C43.9151 189.926 43.5917 189.489 43.7044 189.052C43.7885 188.77 44.0135 188.559 44.2524 188.376C46.2213 186.812 48.3728 185.502 50.4401 184.065C54.096 181.514 57.5413 178.711 60.8461 175.752C72.0394 165.805 80.6456 154.083 85.7645 141.839C89.5188 132.865 91.0095 123.524 91.2766 114.549C91.3191 113.013 91.3473 111.477 91.3047 109.984C91.2909 108.11 91.2063 106.25 91.1079 104.419C90.9813 102.291 90.7988 100.206 90.5876 98.163C90.2923 95.4296 89.9551 92.7951 89.5752 90.2591C94.1313 89.2025 98.6735 87.7932 103.201 86.5537C108.039 85.2433 112.848 83.8905 117.644 82.4253C119.092 81.9747 120.54 81.5236 121.975 81.0448C123.479 80.5516 124.97 80.0303 126.447 79.4951C127.937 78.9593 129.414 78.396 130.876 77.8044C132.339 77.2122 133.787 76.5925 135.235 75.9301C136.67 75.2683 138.09 74.5778 139.482 73.8452C140.185 73.4787 140.875 73.0983 141.549 72.7179C141.887 72.5205 142.239 72.3237 142.576 72.1263C142.745 72.0274 142.913 71.9289 143.082 71.8305C143.11 71.8162 143.518 71.5768 143.504 71.5624C143.237 71.3512 142.956 71.1964 142.674 70.999C142.464 70.858 142.266 70.6894 142.056 70.5622C141.001 69.8999 139.932 69.2381 138.877 68.5757C136.529 67.1106 134.195 65.6875 131.692 64.5182C130.398 63.9122 129.104 63.3063 127.937 62.4891C127.923 62.4753 128.893 61.8693 128.978 61.8268C129.202 61.7001 129.442 61.5735 129.681 61.4464C130.019 61.2777 130.314 61.0378 130.637 60.8266C131.523 60.277 132.451 59.7982 133.351 59.263C134.392 58.6289 135.46 58.0368 136.543 57.4734C138.273 59.1923 140.635 60.277 142.857 61.1931C145.29 62.1933 148.426 62.0385 150.971 61.6299C154.881 60.9958 158.565 59.3471 161.94 57.2478C169.646 52.4577 173.541 46.0471 174.188 37.0864C174.427 33.7614 174.666 30.6336 176.339 28.4075C178.308 25.801 182.175 24.3779 185.916 22.9972C188.826 21.9405 191.794 20.6444 194.705 19.3763C196.237 18.7282 197.756 18.0519 199.289 17.432C199.922 17.1643 200.808 16.6148 201.511 16.6007C202.678 16.5726 204.576 17.5306 205.532 18.1505C206.756 18.9396 206.278 20.6866 205.954 21.856C204.253 27.9144 203.114 34.029 201.005 39.9888C197.784 49.034 193.327 57.6 187.758 65.4199C183.61 71.2528 177.324 75.1694 171.952 79.7063C164.738 85.8067 160.688 95.162 160.646 105.687C160.646 105.786 160.491 105.828 160.407 105.841C160.069 105.912 159.717 105.87 159.38 105.841C158.818 105.799 158.255 105.729 157.707 105.645C156.99 105.546 156.272 105.419 155.555 105.306C154.74 105.165 153.924 105.01 153.108 104.856C152.251 104.686 151.407 104.517 150.549 104.348C149.72 104.179 148.89 104.01 148.046 103.841C147.287 103.686 146.541 103.531 145.782 103.362C145.164 103.235 144.559 103.094 143.94 102.968C143.603 102.897 143.279 102.798 142.956 102.756C142.941 102.756 142.913 102.756 142.885 102.798C142.871 102.869 142.913 102.953 142.941 103.024C143.026 103.165 143.082 103.305 143.195 103.418C143.701 103.968 144.193 104.532 144.784 105.01C148.552 108.025 153.052 109.857 157.848 110.378C159.225 110.533 160.618 110.604 162.01 110.576C178.463 110.336 192.16 101.812 203.873 90.9634C208.809 86.3983 213.928 81.6789 220.115 81.073C224.094 80.6782 228.088 82.1998 230.268 84.9193C232.279 87.4272 233.109 91.2028 232.645 95.5988ZM177.338 102.968C173.527 104.306 169.547 105.151 165.54 105.517C165.638 96.5425 169.041 88.5822 175.102 83.4819C175.692 82.9887 176.297 82.5243 176.873 82.045C174.919 88.4694 175.074 95.9653 177.338 102.968ZM234.107 81.8901C233.094 80.6218 231.843 79.5653 230.437 78.6917C230.887 78.3396 231.323 77.973 231.745 77.6208C235.654 74.6624 239.423 71.9853 243.923 71.182C244.302 71.1118 244.668 71.0698 245.019 71.098C245.005 71.182 244.991 71.2948 244.949 71.4358C244.19 75.3103 242.769 78.1847 240.393 80.7346L236.146 85.2715C235.626 84.1022 234.965 82.9611 234.107 81.8901ZM244.625 87.3144C243.318 91.696 240.59 94.6832 237.299 97.7124C237.383 97.191 237.468 96.6553 237.524 96.1058C237.693 94.556 237.735 92.7387 237.538 90.8506C238.213 90.3011 238.888 89.7239 239.634 89.1461C240.871 88.1598 242.179 87.1596 243.332 85.9759C244.021 85.2854 244.597 84.5672 245.16 83.8341C245.16 84.9475 244.977 86.1169 244.625 87.3144Z" fill="#1D445C"/>
<path d="M57.1758 149.912C57.063 150.124 56.9369 150.335 56.9508 150.574C56.9646 150.814 57.1476 151.054 57.3865 151.039C57.5131 151.026 57.6115 150.955 57.7099 150.885C60.0161 149.222 62.3506 147.531 64.2493 145.418C66.1475 143.305 67.6381 140.698 67.9472 137.866C67.9898 137.415 67.9334 136.851 67.5115 136.711C67.3147 136.64 67.1035 136.697 66.9066 136.739C65.6272 137.091 63.7567 137.768 62.8709 138.81C62.1399 139.669 62.0973 140.811 61.6755 141.882C60.5364 144.742 58.6803 147.249 57.1758 149.912Z" fill="#1D445C"/>
<path d="M66.4287 147.926C66.2744 148.194 66.1196 148.49 66.1334 148.786C66.1478 149.095 66.3867 149.42 66.6963 149.392C66.8506 149.377 66.9916 149.279 67.1182 149.194C70.1272 147.024 73.1505 144.827 75.6398 142.079C78.1285 139.332 80.0554 135.936 80.4491 132.245C80.5055 131.668 80.4347 130.921 79.8867 130.738C79.6335 130.653 79.3521 130.709 79.0988 130.78C77.4395 131.245 74.9929 132.104 73.8395 133.471C72.8835 134.598 72.8414 136.077 72.2929 137.472C70.8023 141.191 68.3976 144.46 66.4287 147.926Z" fill="#1D445C"/>
<path d="M106.605 99.9099C106.521 100.036 106.366 100.107 106.212 100.177C102.71 101.713 99.1663 103.249 95.37 103.996C93.9916 104.278 92.5711 104.447 91.1512 104.475C89.4919 104.517 87.8044 104.362 86.2009 103.967C86.1169 102.263 85.9759 100.572 85.7791 98.8953C86.5106 98.4447 87.2841 98.0925 88.0294 97.9654C88.9152 97.8244 89.7728 97.9936 90.6309 98.233C91.2635 98.4022 91.8965 98.5713 92.5711 98.6698C96.8181 99.2901 101.093 98.8671 105.326 98.9379C105.649 98.9379 106.015 98.9517 106.282 99.1209C106.577 99.2757 106.774 99.6561 106.605 99.9099Z" fill="#1D445C"/>
<path d="M100.727 110.97C100.642 111.097 100.487 111.167 100.333 111.238C97.3938 112.506 94.4407 113.802 91.305 114.62C90.7002 114.803 90.0958 114.929 89.4771 115.056C88.4642 115.267 87.4236 115.408 86.3691 115.479C86.4676 113.534 86.4814 111.604 86.4394 109.66C86.524 109.674 86.6224 109.688 86.7064 109.702C88.2535 109.928 89.8005 110.012 91.3471 110.026C94.047 110.069 96.7613 109.913 99.4612 109.956C99.7846 109.956 100.136 109.97 100.417 110.139C100.712 110.308 100.895 110.702 100.727 110.97Z" fill="#1D445C"/>
<path d="M182.837 193.871C181.5 195.322 180.136 196.731 178.702 198.07C177.071 197.351 175.524 196.519 174.005 195.618C172.74 194.871 171.502 194.082 170.251 193.293C170.124 193.209 169.97 193.11 169.913 192.969C169.787 192.687 170.026 192.349 170.349 192.222C170.659 192.11 170.996 192.138 171.348 192.194C173.12 192.462 174.905 192.786 176.663 193.096C178.702 193.448 180.756 193.772 182.837 193.871Z" fill="#1D445C"/>
<path d="M175.074 201.282C173.147 202.888 171.151 204.396 169.055 205.818C167.902 205.283 166.777 204.649 165.737 203.945C165.287 203.649 164.837 203.325 164.415 203.015C161.743 201 159.409 198.619 157.117 196.252C157.004 196.139 156.892 196.026 156.85 195.858C156.807 195.561 157.13 195.266 157.454 195.224C157.792 195.195 158.129 195.308 158.439 195.435C161.996 196.816 165.413 198.619 169.112 199.718C169.534 199.845 169.955 199.957 170.406 200.07C171.966 200.465 173.598 200.352 174.92 201.169C174.948 201.197 175.018 201.226 175.074 201.282Z" fill="#1D445C"/>
</svg>';
		if ( ! $show_icon_only ) {
			$output .= '<span>' . esc_html( $string ) . '</span>';
		}
		$output .= '</a>';

		echo $output;
	}

	/**
	 * Check if we should rewrite the urls.
	 *
	 * @return bool If we can replace the image.
	 */
	public function should_replace() {

		if ( apply_filters( 'optml_should_replace_page', false ) ) {
			return false;
		}
		if ( apply_filters( 'optml_force_replacement', false ) === true ) {
			return true;
		}

		if ( is_customize_preview() && $this->settings->get( 'offload_media' ) !== 'enabled' ) {
			return false;
		}

		if ( ( is_admin() && ! self::is_ajax_request() ) || ! $this->settings->is_connected() || ! $this->settings->is_enabled() ) {
			return false; // @codeCoverageIgnore
		}

		if ( array_key_exists( 'preview', $_GET ) && ! empty( $_GET['preview'] ) && ! $this->settings->is_offload_enabled() ) {
			return false; // @codeCoverageIgnore
		}

		if ( array_key_exists( 'optml_off', $_GET ) && 'true' === $_GET['optml_off'] ) {
			return false; // @codeCoverageIgnore
		}
		if ( array_key_exists( 'elementor-preview', $_GET ) && ! empty( $_GET['elementor-preview'] ) ) {
			return false; // @codeCoverageIgnore
		}
		if ( array_key_exists( 'ct_builder', $_GET ) && ! empty( $_GET['ct_builder'] ) ) {
			return false; // @codeCoverageIgnore
		}
		if ( array_key_exists( 'et_fb', $_GET ) && ! empty( $_GET['et_fb'] ) ) {
			return false; // @codeCoverageIgnore
		}
		if ( array_key_exists( 'tve', $_GET ) && $_GET['tve'] === 'true' ) {
			return false; // @codeCoverageIgnore
		}
		if ( array_key_exists( 'trp-edit-translation', $_GET ) && ( $_GET['trp-edit-translation'] === 'true' || $_GET['trp-edit-translation'] === 'preview' ) ) {
			return false; // @codeCoverageIgnore
		}
		if ( array_key_exists( 'context', $_GET ) && $_GET['context'] === 'edit' ) {
			return false; // @codeCoverageIgnore
		}
		// avada
		if ( array_key_exists( 'fb-edit', $_GET ) && ! empty( $_GET['fb-edit'] ) ) {
			return false; // @codeCoverageIgnore
		}
		if ( array_key_exists( 'builder', $_GET ) && ! empty( $_GET['builder'] )
			&& array_key_exists( 'builder_id', $_GET ) && ! empty( $_GET['builder_id'] ) ) {
			return false; // @codeCoverageIgnore
		}
		// Motion.page iFrame & builder
		if ( ( array_key_exists( 'motionpage_iframe', $_GET ) && $_GET['motionpage_iframe'] === 'true' ) || ( array_key_exists( 'page', $_GET ) && $_GET['page'] === 'motionpage' ) ) {  // phpcs:ignore WordPress.PHP.StrictComparisons.LooseComparison
			return false; // @codeCoverageIgnore
		}
		/**
		 * Disable replacement on POST request and when user is logged in, but allows for sample image call widget in dashboard
		 */
		if (
			isset( $_SERVER['REQUEST_METHOD'] ) &&
			$_SERVER['REQUEST_METHOD'] === 'POST' &&
			is_user_logged_in()
			&& ( ! isset( $_GET['quality'] ) || ! current_user_can( 'manage_options' ) )
		) {
			return false; // @codeCoverageIgnore
		}
		if ( class_exists( 'FLBuilderModel', false ) ) {
			$post_data = FLBuilderModel::get_post_data();
			if ( isset( $_GET['fl_builder'] ) || isset( $post_data['fl_builder'] ) ) {
				return false;
			}
		}
		$filters = $this->settings->get_filters();

		return Optml_Filters::should_do_page( $filters[ Optml_Settings::FILTER_TYPE_OPTIMIZE ][ Optml_Settings::FILTER_URL ], $filters[ Optml_Settings::FILTER_TYPE_OPTIMIZE ][ Optml_Settings::FILTER_URL_MATCH ] );
	}

	/**
	 * Check if we are in a ajax contex where we should enable replacement.
	 *
	 * @return bool Is ajax request?
	 */
	public static function is_ajax_request() {
		if ( apply_filters( 'optml_force_replacement_on', false ) === true ) {

			return true;
		}
		if ( ! function_exists( 'is_user_logged_in' ) ) {
			return false;
		}
		// Disable for logged in users to avoid unexpected results.
		if ( is_user_logged_in() ) {
			return false;
		}

		if ( ! function_exists( 'wp_doing_ajax' ) ) {
			return false;
		}
		if ( ! wp_doing_ajax() ) {
			return false;
		}
		if ( isset( $_REQUEST['action'] ) && strpos( $_REQUEST['action'], 'wpmdb' ) !== false ) {
			return false;
		}

		return true;
	}

	/**
	 * Register frontend replacer hooks.
	 */
	public function register_hooks() {

		do_action( 'optml_replacer_setup' );
		if ( $this->settings->get( 'native_lazyload' ) === 'disabled' ) {
			add_filter( 'wp_lazy_loading_enabled', '__return_false' );
		}
		add_filter( 'the_content', [ $this, 'process_images_from_content' ], PHP_INT_MAX );
		/**
		 * When we have to process cdn images, i.e MIRROR is defined,
		 * we need this as late as possible for other replacers to occur.
		 * Otherwise, we can hook first to avoid any other plugins to take care of replacement.
		 */
		add_action(
			self::is_ajax_request() ? 'init' : 'template_redirect',
			[
				$this,
				'process_template_redirect_content',
			],
			defined( 'OPTML_SITE_MIRROR' ) ? PHP_INT_MAX : PHP_INT_MIN
		);
		add_action( 'template_redirect', [ $this, 'register_after_setup' ] );
		add_action( 'rest_api_init', [ $this, 'process_template_redirect_content' ], PHP_INT_MIN );
		add_action( 'shutdown', [ $this, 'close_buffer' ] );
		foreach ( self::$loaded_compatibilities as $registered_compatibility ) {
			$registered_compatibility->register();
		}
	}

	/**
	 * Run after Optimole is fully setup.
	 */
	public function register_after_setup() {
		do_action( 'optml_after_setup' );
	}

	/**
	 * Check if we should load the profiler.
	 *
	 * @param bool $default_value Default value.
	 *
	 * @return bool Should load the profiler.
	 */
	public static function should_load_profiler( $default_value = false ) {
		return ! $default_value && apply_filters( 'optml_page_profiler_disable', false ) === false;
	}
	/**
	 * Filter raw HTML content for urls.
	 *
	 * @param string $html HTML to filter.
	 * @param bool   $partial If this is a partial content replacement and not a full page. It matters when we are are doing full page optimization like viewport lazyload.
	 *
	 * @return mixed Filtered content.
	 */
	public function replace_content( $html, $partial = false ) {

		if ( defined( 'REST_REQUEST' ) && REST_REQUEST && is_user_logged_in() && ( apply_filters( 'optml_force_replacement', false ) !== true ) ) {
			return $html;
		}
		if ( self::should_load_profiler( $partial ) ) {
			$profile_id = Profile::generate_id( $html );
			$should_show_comment = false;
			// We disable the optimizer for logged in users.
			if ( ! is_user_logged_in() || ! apply_filters( 'optml_force_page_profiler', false ) !== true ) {
				$js_optimizer = Optml_Admin::get_optimizer_script( false );

				if ( ! $this->page_profiler->exists_all( $profile_id ) ) {
					$missing = $this->page_profiler->missing_devices( $profile_id );
					$time = time();
					$hmac = wp_hash( $profile_id . $time . $this->get_current_url(), 'nonce' );
					$js_optimizer = str_replace(
						[ Profile::PLACEHOLDER, Profile::PLACEHOLDER_MISSING, Profile::PLACEHOLDER_TIME, Profile::PLACEHOLDER_HMAC, Profile::PLACEHOLDER_URL ],
						[ $profile_id, implode( ',', $missing ), strval( $time ), $hmac, $this->get_current_url() ],
						$js_optimizer
					);
					$html = str_replace( Optml_Admin::get_optimizer_script( true ), $js_optimizer, $html );
					if ( ! headers_sent() ) {
						header( 'Cache-Control: max-age=300' ); // Attempt to cache the page just for 5 mins until the optimizer is done. Once the optimizer is done, the page will load optimized.
					}
				} else {
					$should_show_comment = isset( $_GET['optml_debug'] ) && $_GET['optml_debug'] === 'true';
				}
			}

			Profile::set_current_profile_id( $profile_id );
			$this->page_profiler->set_current_profile_data();
			if ( $should_show_comment ) {
				$html = str_replace( '</html>', '</html>' . $this->page_profiler->get_current_profile_html_comment(), $html );
			}
		}
		if ( ! $partial ) {
			$html = $this->add_html_class( $html );
		}

		$html = $this->process_images_from_content( $html );

		if ( $this->settings->get( 'video_lazyload' ) === 'enabled' ) {
			$html = apply_filters( 'optml_video_replace', $html );
			if ( Optml_Lazyload_Replacer::found_iframe() === true ) {
				if ( strpos( $html, Optml_Lazyload_Replacer::IFRAME_TEMP_COMMENT ) !== false ) {
					$html = str_replace( Optml_Lazyload_Replacer::IFRAME_TEMP_COMMENT, Optml_Lazyload_Replacer::IFRAME_PLACEHOLDER_CLASS, $html );
				} else {
					$html = preg_replace( '/<head>(.*)<\/head>/ism', '<head> $1' . Optml_Lazyload_Replacer::IFRAME_PLACEHOLDER_STYLE . '</head>', $html );
				}
			}
		}

		if ( $this->settings->is_lazyload_type_viewport() && ! $partial ) {
			$personalized_bg_css = Lazyload::get_current_personalized_css();
			if ( OPTML_DEBUG ) {
				do_action( 'optml_log', 'viewport_bgselectorsdata: ' . print_r( $personalized_bg_css, true ) );
			}

			if ( ! empty( $personalized_bg_css ) && ( $start_pos = strpos( $html, Lazyload::MARKER ) ) !== false ) { // phpcs:ignore Generic.CodeAnalysis.AssignmentInCondition.Found
				// We replace the general bg css with the personalized one.
				if ( ( $end_pos = strpos( $html, Lazyload::MARKER, $start_pos + strlen( Lazyload::MARKER ) ) ) !== false ) { // phpcs:ignore Generic.CodeAnalysis.AssignmentInCondition.Found
					$html = substr_replace(
						$html,
						$personalized_bg_css,
						$start_pos,
						$end_pos + strlen( Lazyload::MARKER ) - $start_pos
					);
				}
			}
		}
		if ( ! $partial ) {
			// WE need this last since during bg personalized CSS we collect preload urls
			if ( Links::get_links_count() > 0 ) {
				if ( OPTML_DEBUG ) {
					do_action( 'optml_log', 'preload_links: ' . print_r( Links::get_links(), true ) );
				}
				$html = str_replace( Optml_Admin::get_preload_links_placeholder(), Links::get_links_html(), $html );
			} else {
				$html = str_replace( Optml_Admin::get_preload_links_placeholder(), '', $html );
			}
		}

		$html = apply_filters( 'optml_url_pre_process', $html );

		$html = $this->process_urls_from_content( $html );

		$html = apply_filters( 'optml_url_post_process', $html );
		if ( self::should_load_profiler( $partial ) ) {
			Profile::reset_current_profile();
		}
		return $html;
	}

	/**
	 * Get the current url.
	 *
	 * @return string The current url.
	 */
	private function get_current_url() {
		global $wp;
		return home_url( add_query_arg( [], $wp->request ) );
	}
	/**
	 * Adds a filter that allows adding classes to the HTML tag.
	 *
	 * @param string $content The HTML content.
	 *
	 * @return mixed
	 */
	public function add_html_class( $content ) {
		if ( empty( $content ) ) {
			return $content;
		}

		$additional_html_classes = apply_filters( 'optml_additional_html_classes', [] );

		if ( ! $additional_html_classes ) {
			return $content;
		}

		if ( preg_match( '/<html.*>/ismU', $content, $matches, PREG_OFFSET_CAPTURE ) === 1 ) {

			$add_classes = implode( ' ', $additional_html_classes );
			foreach ( $matches as $match ) {
				if ( strpos( $match[0], 'class' ) !== false ) {
					$new_tag = str_replace( [ 'class="', "class='" ], [ 'class="' . $add_classes, "class='" . $add_classes ], $match[0] );
				} else {
					$new_tag = str_replace( 'html ', 'html class="' . $add_classes . '" ', $match[0] );
				}

				$content = str_replace( $match[0], $new_tag, $content );
			}
		}

		return $content;
	}

	/**
	 * Adds a filter with detected images tags and the content.
	 *
	 * @param string $content The HTML content.
	 *
	 * @return mixed
	 */
	public function process_images_from_content( $content ) {
		if ( self::should_ignore_image_tags() ) {
			return $content;
		}
		$images = self::parse_images_from_html( $content );
		if ( empty( $images ) ) {
			return $content;
		}
		return apply_filters( 'optml_content_images_tags', $content, $images );
	}

	/**
	 * Check if we are on a amp endpoint.
	 *
	 * IMPORTANT: This needs to be  used after parse_query hook, otherwise will return false positives.
	 *
	 * @return bool
	 */
	public static function should_ignore_image_tags() {

		// Ignore image tag replacement in feed context as we don't need it.
		if ( is_feed() ) {
			return true;
		}

		// Ignore image tags replacement in amp context as they are not available.
		if ( function_exists( 'is_amp_endpoint' ) ) {
			return is_amp_endpoint();
		}

		if ( function_exists( 'ampforwp_is_amp_endpoint' ) ) {
			return ampforwp_is_amp_endpoint();
		}

		return apply_filters( 'optml_should_ignore_image_tags', false ) === true;
	}

	/**
	 * Match all images and any relevant <a> tags in a block of HTML.
	 *
	 * @param string $content Some HTML.
	 *
	 * @return array An array of $images matches, where $images[0] is
	 *         an array of full matches, and the link_url, img_tag,
	 *         and img_url keys are arrays of those matches.
	 */
	public static function parse_images_from_html( $content ) {
		$images = [];

		if ( OPTML_DEBUG ) {
			do_action( 'optml_log', 'Content to parse images from: ' . $content );
		}
		$regex = '/(?:<a[^>]+?href=["|\'](?P<link_url>[^\s]+?)["|\'][^>]*?>\s*)?(?P<img_tag>(?:<noscript\s*>\s*)?<img[^>]*?\s?(?:' . implode( '|', array_merge( [ 'src' ], Optml_Tag_Replacer::possible_src_attributes() ) ) . ')=["\'\\\\]*?(?P<img_url>[' . Optml_Config::$chars . ']{10,}).*?>(?:\s*<\/noscript\s*>)?){1}(?:\s*<\/a>)?/ismu';

		if ( preg_match_all( $regex, $content, $images, PREG_OFFSET_CAPTURE ) ) {
			if ( OPTML_DEBUG ) {
				do_action( 'optml_log', 'Image tags parsed: ' . print_r( $images, true ) );
			}
			foreach ( $images as $key => $unused ) {
				// Simplify the output as much as possible, mostly for confirming test results.
				if ( is_numeric( $key ) && $key > 0 ) {
					unset( $images[ $key ] );
					continue;
				}

				$is_no_script = false;
				foreach ( $unused as $url_key => $url_value ) {
					if ( $key === 'img_url' ) {
						$images[ $key ][ $url_key ] = rtrim( $url_value[0], '\\' );
						continue;
					}
					$images[ $key ][ $url_key ] = $url_value[0];

					if ( $key === 0 ) {
						$images['in_no_script'][ $url_key ] = false;

						// Check if we are in the noscript context.
						if ( $is_no_script === false ) {
							$is_no_script = strpos( $images[0][ $url_key ], '<noscript' ) !== false ? true : false;
						}
						if ( $is_no_script ) {
							$images['in_no_script'][ $url_key ] = true;
							$is_no_script                    = strpos( $images[0][ $url_key ], '</noscript' ) !== false ? false : true;
						}
					}
				}
			}

			return $images;
		}

		return [];
	}

	/**
	 * Process url replacement from raw html strings.
	 *
	 * @param string $html Raw html.
	 *
	 * @return string Processed string.
	 */
	public function process_urls_from_content( $html ) {
		$extracted_urls = $this->extract_urls_from_content( $html );
		if ( OPTML_DEBUG ) {
			do_action( 'optml_log', 'Extracted image urls from content: ' . print_r( $extracted_urls, true ) );
		}
		return $this->do_url_replacement( $html, $extracted_urls );
	}

	/**
	 * Method to extract assets from content.
	 *
	 * @param string $content The HTML content.
	 *
	 * @return array
	 */
	public function extract_urls_from_content( $content ) {
		$extensions = array_keys( Optml_Config::$image_extensions );
		if ( $this->settings->use_cdn() && ! self::should_ignore_image_tags() ) {
			$extensions = array_merge( $extensions, array_keys( Optml_Config::$assets_extensions ) );
		}
		$regex = '/(?:[(|\s\';",=\]])((?:http|\/|\\\\){1}(?:[' . Optml_Config::$chars . ']{10,}\.(?:' . implode( '|', $extensions ) . ')))(?=(?:http|>|%3F|\?|"|&|,|\s|\'|\)|\||\\\\|}|\[))/Uu';
		preg_match_all(
			$regex,
			$content,
			$urls
		);

		return $this->normalize_urls( $urls[1] );
	}

	/**
	 * Normalize extracted urls.
	 *
	 * @param array $urls Raw urls extracted.
	 *
	 * @return array Normalized array.
	 */
	private function normalize_urls( $urls ) {

		$urls = array_map(
			function ( $value ) {
				$value = str_replace( '&quot;', '', $value );

				return rtrim( $value, '\\";\'' );
			},
			$urls
		);
		$urls = array_unique( $urls );

		return array_values( $urls );
	}

	/**
	 * Process string content and replace possible urls.
	 *
	 * @param string $html String content.
	 * @param array  $extracted_urls Urls to check.
	 *
	 * @return string Processed html.
	 */
	public function do_url_replacement( $html, $extracted_urls ) {
		$extracted_urls = apply_filters( 'optml_extracted_urls', $extracted_urls );

		if ( empty( $extracted_urls ) ) {
			return $html;
		}
		$slashed_config = addcslashes( Optml_Config::$service_url, '/' );

		$extracted_urls  = array_filter(
			$extracted_urls,
			function ( $value ) use ( $slashed_config ) {
				return strpos( $value, Optml_Config::$service_url ) === false && strpos( $value, $slashed_config ) === false || Optml_Media_Offload::is_not_processed_image( $value ) || $this->tag_replacer->url_has_dam_flag( $value );
			}
		);
		$upload_resource = $this->tag_replacer->get_upload_resource();
		$urls            = array_combine( $extracted_urls, $extracted_urls );

		$urls = array_map(
			function ( $url ) use ( $upload_resource ) {
				$is_slashed  = strpos( $url, '\/' ) !== false;
				$is_relative = strpos(
					$url,
					$is_slashed ?
										addcslashes( $upload_resource['content_path'], '/' ) :
										$upload_resource['content_path']
				) === 0;
				if ( $is_relative ) {
					$url = $upload_resource['content_host'] . $url;
				}

				return apply_filters( 'optml_content_url', $url );
			},
			$urls
		);

		foreach ( $urls as $origin => $replace ) {
			$html = preg_replace( '/(?<![\/|:|\\w])' . preg_quote( $origin, '/' ) . '/m', $replace, $html );
		}

		return $html;
	}

	/**
	 * Init html replacer handler.
	 */
	public function process_template_redirect_content() {
		// Early exit if function was already called, we don't want duplicate ob_start
		if ( self::$ob_started === true ) {
			return;
		}
		self::$ob_started = true;
		// We no longer need this if the handler was started.
		remove_filter( 'the_content', [ $this, 'process_images_from_content' ], PHP_INT_MAX );

		ob_start(
			function ( $content ) {
					/*
					* Wrap the call to replace_content() so that PHP’s output-buffering system
					* does not pass its own second argument ($phase bitmask) to our method.
					*
					* replace_content() expects the second parameter to be a boolean $partial,
					* indicating whether the content is a partial replacement (e.g. for
					* viewport lazy-load) or a full page. If PHP’s $phase integer is passed
					* directly, it would be misinterpreted as $partial and break the logic.
					*
					* This closure filters the call, forwarding only the captured HTML buffer.
					*/
				return $this->replace_content( $content, self::is_ajax_request() );
			}
		);
	}

	/**
	 * Close the buffer and flush the content.
	 */
	public function close_buffer() {
		if ( self::$ob_started && ob_get_length() ) {
			ob_end_flush();
		}
	}
	/**
	 * Throw error on object clone
	 *
	 * The whole idea of the singleton design pattern is that there is a single
	 * object therefore, we don't want the object to be cloned.
	 *
	 * @codeCoverageIgnore
	 * @access public
	 * @return void
	 * @since  1.0.0
	 */
	public function __clone() {
		// Cloning instances of the class is forbidden.
		_doing_it_wrong( __FUNCTION__, esc_html__( 'Cheatin&#8217; huh?', 'optimole-wp' ), '1.0.0' );
	}

	/**
	 * Disable unserializing of the class
	 *
	 * @codeCoverageIgnore
	 * @access public
	 * @return void
	 * @since  1.0.0
	 */
	public function __wakeup() {
		// Unserializing instances of the class is forbidden.
		_doing_it_wrong( __FUNCTION__, esc_html__( 'Cheatin&#8217; huh?', 'optimole-wp' ), '1.0.0' );
	}
}
